import json
import csv
import os
import sys
from CdScriptingNodeHelper import ScriptingResponse

def print_error(*args, **kwargs):
    print(*args, file=sys.stderr, **kwargs)

#
# Scripting Node for Compound Discoverer that calculates difference in Features Molecolar Weigth against m/z
#

              
def main():
    print('CD Scripting Node')
                
    # start in development mode where nodeargs are given explicitely rather than reading it as command line argument
    if sys.argv[1] == '-devel':    
        print(f'Development mode: Current Dir is {os.getcwd()}')
        nodeargs_path = 'node_args.json'        
    else:
        nodeargs_path = sys.argv[1]        

    # parse node args from Compound Discoverer and extract location of Features table
    try:
        with open(nodeargs_path, 'r') as rf:
            nodeargs = json.load(rf) 
            features_path = ''
            response_path = nodeargs['ExpectedResponsePath']
            tables = nodeargs['Tables']
            for table in tables:
                if table['TableName'] == 'Features':                    
                    features_path = table['DataFile']                                    
                    if table['DataFormat'] != 'CSV':
                        print_error(f"Unknown Data Format {table['DataFormat']}")    
                        exit(1)
    except Exception as e: 
        print_error('Could not read Compound Discoverer node args')
        print_error(str(e))
        exit(1)

    if not features_path:
        print_error('Features file not defined in node args.')
        exit(1)


    entries = list()

    
    try:
        with open(features_path, mode='r') as protFile:
            reader = csv.DictReader(protFile, delimiter='\t')
            for row in reader:
                id1 = row['Features WorkflowID']
                id2 = row['Features ID']
                molecular_weight =float(row['Molecular Weight'])
                mz = float(row['mz'])
                z = int(row['Charge'])

                measured_mass = mz * z
                delta = abs(measured_mass - molecular_weight)

                entries.append([id1, id2, delta])


    except Exception as e:
        print_error('Could not process data')
        print_error(e)        
        exit(1)
        
    
    # write data file
    outfilename = 'FeaturesWithDeltaMass.txt'
    (workdir, _ ) = os.path.split(response_path)
    outfile_path = os.path.join(workdir, outfilename)

    # todo: find out whether the fields need to be in quotes
    with open(outfile_path, mode='w', newline='') as features_csv:
        fieldnames = ['Features WorkflowID', 'Features ID', 'Delta Mass']
        writer = csv.DictWriter(features_csv, fieldnames = fieldnames, delimiter = '\t', quoting = csv.QUOTE_ALL)
        writer.writeheader()

        for entry in entries:
            writer.writerow({'Features WorkflowID': entry[0], 'Features ID': entry[1], 'Delta Mass': entry[2]})
                
    # entries for new column in Features table
    response = ScriptingResponse()
    response.add_table('Features', outfile_path)
    response.add_column('Features', 'Features WorkflowID', 'Int', 'WorkflowID')
    response.add_column('Features', 'Features ID', 'Int', 'ID')
    response.add_column('Features', 'Delta Mass', 'Float')

    # save to disk
    response.save(response_path)




if __name__== "__main__" :
    main()