#
# CompoundsFindExpected.R    08/01/2024    Ralf Tautenhahn       ralf.tautenhahn@thermofisher.com
# 
#
# v1.2
# 
#  Creates column "Expected Compounds" in Compounds table, indicating which Compounds have also been detected in Expected Compounds 
#
#
# Requested Tables and Columns:   Compounds: Calc. MW; Merged Features
#  
#
#


getTableIdx <- function(JSON_in, name) {
  TableNames <- sapply (JSON_in$Tables, function(x) x$TableName)
  which(TableNames %in% name)
}

getTable <- function(JSON_in, name) {
  
  TableNames <- sapply (JSON_in$Tables, function(x) x$TableName)
  idx <- TableNames %in% name
  
  if (!any(idx)) 
    stop("Table not found.")
  
  read.table(JSON_in$Tables[[ which(idx) ]]$DataFile, 
             header=TRUE, check.names = FALSE, stringsAsFactors = FALSE)
}

# Read arguments from CD.
args <- commandArgs()

# 6th argument is the name of the JSON file
inputFile <- args[6]

# Open JSON file, find exported files, read into tables
library(rjson)
CD_json_in <- fromJSON(file=inputFile)

Compounds <- getTable(CD_json_in, "Compounds")
MergedFeatures <- getTable(CD_json_in, "Merged Features")
CompoundsMergedFeatures <- getTable(CD_json_in, "ConsolidatedUnknownCompoundItem-ConsolidatedComponentItem")

# for debugging only 
# save.image(file="C:\\TEMP\\CD node Rimage FE.dat")
# load("C:\\TEMP\\CD node Rimage FE.dat")

 res <- logical(nrow(Compounds))
 
 ## loop through all Compounds
 for (ci in 1:nrow(Compounds)) {
   
   cid <- Compounds[ci,"Compounds ID"]
   MFIdx <- which(CompoundsMergedFeatures[,'Compounds ID'] %in% cid)
   MFIDs <- CompoundsMergedFeatures[MFIdx, "Merged Features ID" ]
   MFData <- MergedFeatures[MergedFeatures[,"Merged Features ID"] %in% MFIDs,]

   FEdf <- MFData[, grep("Find Expected Compound.", colnames(MFData))]
   
   #if (all(grepl("No matches found", FEdf)))
   if (all(  apply(FEdf,2, grepl, pattern="No matches found") ))
     res[ci] <- FALSE
   else
     res[ci] <- TRUE
 }
 
 # Calculate Result
 data.output <- cbind(Compounds, "Expected Compounds" = res )
 
 
 # Add new column to JSON structure.
 
 newcolumn <- list()
 newcolumn[[1]] = "Expected Compounds"       ## ColumnName
 newcolumn[[2]] = FALSE      ## IsID
 newcolumn[[3]] = "Boolean"    ## DataType
 newcolumn[[4]] <- list(PositionAfter="Reference Ion", SpecialCellRenderer="330D9522-50CC-41B7-9D45-5E5D8F708103")    ## Options
 
 names(newcolumn) <- c("ColumnName", "IsID", "DataType", "Options") 
 
 CD_json_in$Tables[[1]]$ColumnDescriptions[[length(CD_json_in$Tables[[1]]$ColumnDescriptions) + 1]] <- newcolumn
 
  # Write modified Compounds table to temporary folder.
 Compounds.idx <- getTableIdx(CD_json_in, "Compounds")
 Compounds.datafile <- CD_json_in$Tables[[ Compounds.idx  ]]$DataFile
 resultout <- gsub(".txt", ".out.txt", Compounds.datafile)
 write.table(data.output, file = resultout, sep='\t', row.names = FALSE)
 
 # Write out node_response.json file - use same file as node_args.json but change the pathway input file to the new one
 
 CD_json_in$Tables[[Compounds.idx]]$DataFile = resultout
 jsonOutFile <- CD_json_in$ExpectedResponsePath
 
 # Remove all the other tables in the JSON so that only the new Compounds table is used
 for (j in seq(length(CD_json_in$Tables),2,-1) ) 
   CD_json_in$Tables[j] <- NULL;
 
 responseJSON <- toJSON(CD_json_in, indent=1, method="C")
 
 # responseJSON has incorrect format for the empty Options lists.  Will use a regular expression to find and replace the [\n\n\] with the {}
 
 responseJSON2 <- gsub("\\[\n\n[[:blank:]]+\\]", "{ }", responseJSON)
 
 jsonfileconn <- file(jsonOutFile)
 
 writeLines(responseJSON2, jsonfileconn)
 
 close (jsonfileconn)
 
 